
/************************************************************************
 *																								*
 *		D48 8048/41 Disassembler - Copyright (C) 1995-1996 by					*
 *		Jeffery L. Post																	*
 *		22726 Benner Ave.																	*
 *		Torrance, CA  90505																*
 *																								*
 *		D48.C - Main File																	*
 *																								*
 *		Version 2.0 - 08/20/96															*
 *																								*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.												*
 *																								*
 *	This program is distributed in the hope that it will be useful,		*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.									*
 *																								*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the Free Software				*
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.				*
 *																								*
 ************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<string.h>
#include	"d48.h"
#include	"d48.tbl"

#ifdef	MSDOS
#include	<dos.h>
#include	<alloc.h>
#endif

#ifdef	LINUX
#include	<malloc.h>
#endif

/*********************
 *							*
 *		Prototypes		*
 *							*
 *********************/

void	usage(void);
void	main(int argc, char *argv[]);
void	pass0(void);
void	error(char *str1, char *str2);
void	add_entry(word val, char *symbol, int type);
char	*find_entry(word val, word count, SYM_PTR *table);
struct sym	*get_smem(int type, int size);

struct sym	*sort(struct sym *list, SYM_PTR *array, word count);
struct sym	*sort_by_name(struct sym *list);
struct sym	*sort_by_value(struct sym *list);
struct sym	*merge_by_name(struct sym *a, struct sym *b);
struct sym	*merge_by_value(struct sym *a, struct sym *b);
void	chk_dup_name(struct sym *list, word count);
void	chk_dup_value(struct sym *list, word count);

char	*get_adrs(char *text, word *val);

extern void	pass1(void);
extern void	pass2(void);
extern void	pass3(void);

/***************************
 *									*
 *		Global variables		*
 *									*
 ***************************/

char	src[32], dst[32];				/* file name buffers				*/
char	ctl[32];							/* control file name				*/
char	linebuffer[128];				/* input line buffer				*/
FILE	*fp;								/* source file						*/
byte	hexflag;							/* append hex flag				*/
byte	fileflag;						/* file type flag					*/
byte	flag41;							/* 8041 flag						*/
int	kcnt;								/* output char counter			*/
byte	pdata[PSIZE];					/* program data space			*/
byte	pflag[PSIZE];					/* program flag space			*/
char	string[ASCLIMIT];				/* ascii output buffer			*/
int	asc_cnt;							/* count for ascii data			*/
byte	byte_data[BYTELIMIT];		/* binary data for defb			*/
int	byte_cnt;						/* count for binary data		*/
int	dump;								/* dump just done flag			*/
struct sym	*sym_tab;				/* symbol table pointer			*/
struct sym	*lab_tab;				/* label table pointer			*/
struct sym	*sym_tab_last;			/* last symbol table pointer	*/
struct sym	*lab_tab_last;			/* lastlabel table pointer		*/
word	symbol_count;					/* number of symbols				*/
word	label_count;					/* number of labels				*/
char	defbstr[8];
char	defwstr[8];

SYM_PTR	*sym_val_index;				/* array of pointers				*/
SYM_PTR	*lab_val_index;				/*  for binary search			*/
struct sym	*tail_ptr, *head_ptr;	/* sort pointers					*/

/*********************
 *							*
 *			Code			*
 *							*
 *********************/

void usage(void)
{
	printf("\nUsage: d48 <filename> [options]\n"
		"Options may be entered Unix style (-d) or DOS style (/b)\n"
		"\t-1 disassemble 8041 code.\n"
		"\t-d include address and data in comment field.\n"
		"\t-s change 'db' and 'dw' to 'defb' and 'defw'.\n"
		"\t-b force .bin extension on input file.\n"
		"\t-h force .hex extension on input file.\n"
		"\t   If neither 'b' nor 'h' is specified, D48 will first search\n"
		"\t   for a .hex file, and if not found, then a .bin file.\n\n"
		"Options may be entered in a freeform fashion as long "
		"as a dash (-) or\n"
		"a slash (/) preceeds any option that preceeds the filename."
		"\nExamples:\n"
		"\td48 filename bd\n"
		"\td48 -d filename b\n"
		"\td48 /b -d filename\n");
	exit(GOOD_EXIT);
}

/*********************************
 *											*
 *			The Main Program			*
 *											*
 *********************************/

void main(int argc, char *argv[])
{
	char	c, *inp;
	int	count, line, pc, i, k;

	printf("\nD48 8048/8041 Disassembler V %d.%d\n", VERNO, REVNO);
	printf("Copyright (C) 1996 by J. L. Post\n");
	if (argc < 2)
		usage();
	hexflag = 0;
	fileflag = 0;
	flag41 = 0;
	dump = 0;
	strcpy(defbstr, "db");				/* init define byte and word strings */
	strcpy(defwstr, "dw");

	for (count=0; count < PSIZE; count++)	/* fill code space with	*/
	{
		pdata[count] = NO_DATA;						/* invalid data */
		pflag[count] = PF_INIT;
	}

	fileflag = EITHERFILE;				/* assume search for either file type */
	sym_tab = NULL;						/* no symbols or labels yet */
	lab_tab = NULL;

/*******************************
	find filename in command line
 *******************************/

	for (line=1; line<argc; line++)	/* find first non option string */
	{
		inp = argv[line];
		c = *inp;
		if (c == '?')						/* '?' without preceeding '-' or '/' */
			usage();
		if (c == '-' || c == '/')
		{
			if (*(++inp) == '?')			/* '?' following '-' or '/' */
				usage();
		}
		else
		{										/* assume first found is file name */
			strcpy(src, argv[line]);	/* init filenames */
			strcat(src, ".hex");			/* assume search for .hex first */
			strcpy(dst, argv[line]);
			strcat(dst, ".d48");
			strcpy(ctl, argv[line]);
			strcat(ctl, ".ctl");
			break;
		}
	}

/******************************
	process command line options
 ******************************/

	if (argc > 2)							/* test for options */
	{
		for (count=1; count<argc; count++)
		{
			inp = argv[count];			/* to avoid modifying pointer in argv */
			c = (char) toupper(*inp++);
			while (c)
			{
				if (c == '-' || c == '/')
					c = toupper(*inp++);		/* skip over option specifier */

				if (c == '?')
					usage();

				if (count == line)			/* skip if already identified */
					break;						/* as the file name */

				if (c == 'D')					/* add data in comment field */
					hexflag = 1;

				else if (c == 'B')			/* binary instead of hex file */
				{
					fileflag = BINFILE;
					strcpy(src, argv[line]);
					strcat(src, ".bin");
				}

				else if (c == 'H')			/* force search for hex file */
				{
					fileflag = HEXFILE;
					strcpy(src, argv[line]);
					strcat(src, ".hex");
				}

				else if (c == 'S')			/* change db/dw strings */
				{
					strcpy(defbstr, "defb");
					strcpy(defwstr, "defw");
				}

				else if (c == '1')			/* modify opcode table */
				{
					flag41 = 1;
					strcpy(mnemtbl[1].mnem, defbstr);
					optbl[1] = OPT_INVAL;
					strcpy(mnemtbl[2].mnem, "out dbb,a");
					strcpy(mnemtbl[8].mnem, defbstr);
					optbl[8] = OPT_INVAL;
					strcpy(mnemtbl[34].mnem, "in a,dbb");
					optbl[34] = OPT_NONE;
					strcpy(mnemtbl[117].mnem, defbstr);
					optbl[117] = OPT_INVAL;
					strcpy(mnemtbl[128].mnem, defbstr);
					optbl[128] = OPT_INVAL;
					strcpy(mnemtbl[129].mnem, defbstr);
					optbl[129] = OPT_INVAL;
					strcpy(mnemtbl[134].mnem, "jobf ");
					strcpy(mnemtbl[136].mnem, defbstr);
					optbl[136] = OPT_INVAL;
					strcpy(mnemtbl[144].mnem, "mov sts,a");
					strcpy(mnemtbl[145].mnem, defbstr);
					optbl[145] = OPT_INVAL;
					strcpy(mnemtbl[152].mnem, defbstr);
					optbl[152] = OPT_INVAL;
					strcpy(mnemtbl[214].mnem, "jnibf ");
					bctbl[214] = 2;
					optbl[214] = OPT_PAGE;
					strcpy(mnemtbl[229].mnem, "en dma");
					strcpy(mnemtbl[245].mnem, "en flags");
					printf("8041 disassembly\n");
				}
				c = (char) toupper(*inp++);
			}
		}
	}

/******************
	open source file
 ******************/

	switch (fileflag)
	{
		case EITHERFILE:				/* if no type specified... */
			fp = fopen(src, "r");	/* search for hex file first */
			if (fp == NULL)			/* if not found, search for bin file */
			{
				fileflag = BINFILE;
				strcpy(src, argv[line]);
				strcat(src, ".bin");
				fp = fopen(src, "rb");
				if (fp == NULL)
				{
					printf("* Can't open either '%s.hex' nor '%s.bin' *\n",
						argv[line], argv[line]);
					exit(FILE_ERROR);
				}
				else
					fileflag = BINFILE;
			}
			break;

		case HEXFILE:					/* force hex file */
			fp = fopen(src, "r");
			break;

		case BINFILE:					/* force bin file */
			fp = fopen(src, "rb");
			break;
	}
	if (fp == NULL)					/* if file not found... */
	{
		printf("* Can't open file '%s' *\n", src);
		exit(FILE_ERROR);
	}

/***************************************
	read input file and set up data array
 ***************************************/

	pc = 0;
	printf("\nreading %s\n", src);

	if (fileflag == BINFILE)				/* if binary file... */
	{
		while (!feof(fp))							/* until end of file... */
		{
			i = fread(linebuffer, 1, 128, fp);	/* read a block of data */
			for (k=0; k<i; k++)
			{
				pdata[pc] = linebuffer[k];		/* copy to program space */
				pflag[pc] = PF_DATA;
				pc++;
				if (pc > PSIZE - 1)
				{
					printf("extra data in binary file ignored");
					break;
				}
			}
			if (pc > PSIZE - 1)
				break;
		}
	}

	else											/* else hex file... */
	{
		while (!feof(fp))							/* until end of file */
		{
			*linebuffer = '\0';					/* clear previous junk */
			fgets(linebuffer, 127, fp);		/* read one line */
			inp = linebuffer;						/* local copy of pointer */
							/* get count and address */
			sscanf(inp, "%*c%2x%4x", &i, &pc);
			if (i == 0)								/* done if end of hex record */
				break;
			if (pc > PSIZE-1)
				error("invalid address in ", linebuffer);
			else if (i > 32)
				error("invalid count in ", linebuffer);	/* set limit */
			else
			{
				for (k=0; k<i; k++)
				{
					sscanf(inp+9+k*2, "%2x", &pdata[pc]);		/* set data array */
					pflag[pc] = PF_DATA;
					pc++;
					if (pc > PSIZE)
						error("invalid address in ", linebuffer);
				}
			}
		}
	}
	fclose(fp);

/********************************************************
	Got the program in data array, now let's go to work...
 ********************************************************/

	symbol_count = 0;
	label_count = 0;

	pass0();								/* read control file					*/
	pass1();								/* find internal references		*/
	pass2();								/* disassemble to output file		*/
	pass3();								/* generate equ's for references	*/
	printf("\nDone");					/* go bye-bye							*/
	exit(GOOD_EXIT);
}								/*  End of Main */

/************************************************************************
 *																								*
 *		Read control file, if it exists, and flag areas as code, text,		*
 *		or whatever. Also handle labels, symbols, etc.							*
 *																								*
 *		Some strange things happen here with oring and anding on the		*
 *		flag bits; this is so the user can define something as code,		*
 *		data, or whatever, and assign a label for the same location.		*
 *		We should handle the result intelligently regardless of the			*
 *		order in which such things are found in the control file.			*
 *		Defining a location as code sets the PF_NOINIT bit, which			*
 *		might seem strange at first, but this is to prevent pass 2 from	*
 *		skipping over multiple NOPs if the user specifically defines		*
 *		them as code.																		*
 *																								*
 ************************************************************************/

void pass0(void)
{
	char	*text, func, c, *ltext;
	word	start, stop, code, temp;
	int	i;
	FILE	*fpc;

	fpc = fopen(ctl, "r");
	if (fpc != NULL)			/* if control file exists... */
	{
		printf("\nReading control file   ");
		while (!feof(fpc))					/* until end of file... */
		{
			start = stop = 0;
			*linebuffer = '\0';				/* clear previous line */
			fgets(linebuffer, 127, fpc);	/* read one line */
			text = &linebuffer[1];
			text = get_adrs(text, &start);
			while (1)
			{
				c = *text++;
				if (c != ' ' && c != '\t')	/* skip whitespace */
					break;
			}
			if (c == '\n' || c == ';')		/* if only one numeric... */
				--text;							/* back up to newline */
			func = c;							/* save operator */
			ltext = text;
			--ltext;
			text = get_adrs(text, &stop);
			if (func == '+')					/* check for valid operator */
				stop += (start - 1);
			else if (func == '-' && !stop)
				stop = start;

			switch (toupper(linebuffer[0]))
			{
				case 'A':								/* address */
					if (start < PSIZE && stop < PSIZE)
					{
						do
						{						/* get address to reference */
							code = ((word) pdata[start]) << 8;
							temp = pdata[start + 1] & 0xff;
							code |= temp;
							if (code < PSIZE)
							{
								pflag[code] |= PF_REF;		/* flag referenced address */
								pflag[code] &= ~(PF_CLREF | PF_SPLIT);
								pflag[start] |= PF_ADRS;		/* set flags to adrs code */
								pflag[start++] &=
									~(PF_NOINIT | PF_WORD | PF_BYTE | PF_ASCII | PF_SPLIT);
								pflag[start] |= PF_ADRS;
								pflag[start++] &=
									~(PF_NOINIT | PF_WORD | PF_BYTE | PF_ASCII | PF_SPLIT);
							}
							else
							{
								pflag[start++] |= PF_WORD;
								pflag[start++] |= PF_WORD;
							}
						} while (start < stop);
					}
					else
						printf("\rinvalid address specified: %x, %x\n",
									start, stop);
					break;

				case 'B':								/* byte binary */
					if (start < PSIZE && stop < PSIZE)
					{
						do
						{
							pflag[start] |= PF_BYTE;
							pflag[start++] &=
								~(PF_NOINIT | PF_ADRS | PF_WORD | PF_ASCII | PF_SPLIT);
						} while (start <= stop);
					}
					else
						printf("\rinvalid address specified: %x, %x\n",
									start, stop);
					break;

				case 'C':								/* code */
					if (start < PSIZE && stop < PSIZE)
					{
						do
						{
							pflag[start] |= PF_NOINIT;	/* to catch 00 or FF bytes */
							pflag[start++] &=
								~(PF_ADRS | PF_WORD | PF_BYTE | PF_ASCII | PF_SPLIT);
						} while (start <= stop);
					}
					else
						printf("\rinvalid address specified: %x, %x\n",
									start, stop);
					break;

				case 'I':								/* ignore */
					if (start < PSIZE && stop < PSIZE)
					{
						do
						{
							pflag[start++] = PF_INIT;
						} while (start <= stop);
					}
					else
						printf("\rinvalid address specified: %x, %x\n",
									start, stop);
					break;

				case 'L':								/* label */
					if (start < PSIZE && stop < PSIZE)
					{
						pflag[start] |= PF_REF;			/* flag reference */
						pflag[start] &= ~(PF_CLREF | PF_SPLIT);
						if (isalnum(*ltext))
							add_entry(start, ltext, LABEL);
					}
					else
						printf("\rinvalid address specified: %x, %x\n",
									start, stop);
					break;

				case 'R':								/* register name */
					if (start > 15)
					{
						printf("\rInvalid register address: 0x%x in '%s'\n",
						linebuffer[0], linebuffer);
					}
					else
					{
						for (stop=0; stop<7; stop++)	/* transfer register name */
						{
							func = *ltext++;
							if (isalnum(func))
								rbname[start].dent[stop] = func;
							else
							{
								rbname[start].dent[stop] = '\0';
								break;
							}
						}
					}
					if (stop >= 7)
						rbname[start].dent[7] = '\0';	/* null terminate reg name */
					break;

				case 'S':								/* symbol */
					add_entry(start, ltext, SYMBOL);
					break;

				case 'T':								/* ascii text */
					do
					{
						pflag[start] |= PF_ASCII;
						pflag[start++] &=
							~(PF_NOINIT | PF_ADRS | PF_WORD | PF_BYTE | PF_SPLIT);
					} while (start <= stop);
					break;

				case 'W':								/* word binary */
					if (start < PSIZE && stop < PSIZE)
					{
						do
						{
							pflag[start] |= PF_WORD;
							pflag[start++] &=
								~(PF_NOINIT | PF_ADRS | PF_BYTE | PF_ASCII | PF_SPLIT);
						} while (start <= stop);
					}
					else
						printf("\rinvalid address specified: %x, %x\n",
									start, stop);
					break;

				case 0x00:								/* ignore empty lines */
				case '\n':
				case '\r':
				case '\t':
				case ' ':
				case ';':								/* ignore commented out lines */
					break;

				default:						/* somebody didn't read the docs... */
					printf("\nUnknown control code: 0x%02x in '%s'\n",
						linebuffer[0], linebuffer);
					break;
			}
		}
		if (label_count || symbol_count)	/* set up tail node for sort */
		{
			tail_ptr = (struct sym *) malloc(sizeof(struct sym) + 12);
			tail_ptr->next = tail_ptr;
			tail_ptr->name[0] = 0xfe;		/* set max values for sort */
			tail_ptr->name[1] = 0;
			tail_ptr->val = 0xffff;
		}
		if (label_count)					/* if labels encountered... */
		{
			printf("\r                        ");
			printf("\rSorting labels   ");
			lab_tab_last->next = tail_ptr;	/* set up pointer array for sort */
			lab_val_index = malloc(sizeof(SYM_PTR) * label_count);
			if (lab_val_index == NULL)
			{
				printf("\nINTERNAL ERROR! - No memory for label pointers");
				exit(MEM_ERROR);
			}
			lab_tab = sort(lab_tab, lab_val_index, label_count);
		}
		if (symbol_count)					/* if symbols encountered... */
		{
			printf("\r                        ");
			printf("\rSorting symbols   ");
			sym_tab_last->next = tail_ptr;
			sym_val_index = malloc(sizeof(SYM_PTR) * symbol_count);
			if (sym_val_index == NULL)
			{
				printf("\nINTERNAL ERROR! - No memory for symbol pointers");
				exit(MEM_ERROR);
			}
			sym_tab = sort(sym_tab, sym_val_index, symbol_count);
		}
		fclose(fpc);
	}
	printf("\r                        ");
}

/*********************************************************
 *																			*
 *		Get hexadecimal number from line in control file.	*
 *		Return updated character pointer.						*
 *																			*
 *********************************************************/

char *get_adrs(char *text, word *val)
{
	word	result, start;
	char	c;

	result = start = 0;
	c = toupper(*text);
	while (c)
	{
		if (c == ';')			/* beginning of comment, ignore all else */
			break;
		if (c == '\n')			/* necessary because isspace() includes \n */
			break;
		if (isspace(c))		/* skip leading whitespace */
		{
			text++;
			if (start)			/* if result already begun... */
				break;
		}
		else if (!isxdigit(c))		/* done if not hexadecimal character */
			break;
		else
		{
			start = 1;			/* flag beginning of result conversion */
			c = (c > '9') ? c - 0x37 : c - 0x30;
			result <<= 4;
			result |= ((word) c & 0xf);
			text++;
		}
		c = toupper(*text);
	}
	*val = result;				/* pass number back to caller */
	return(text);				/* and return updated text pointer */
}

void error(char *str1, char *str2)			/* fatal error trap */
{
	printf("\n%s%s", str1, str2);
	exit(FILE_ERROR);
}

/************************************************************
 *																				*
 *					Sort label or symbol table							*
 *																				*
 *	First sort by name so that we can check for duplicates,	*
 *	then sort by value, check for duplicates, and set up		*
 *	pointer array for binary search.									*
 *																				*
 ************************************************************/

struct sym *sort(struct sym *list, SYM_PTR *array, word count)
{
	word	i;
	struct sym	*sptr, *temp;

	sptr = sort_by_name(list);
	chk_dup_name(sptr, count);
	sptr = sort_by_value(sptr);
	chk_dup_value(sptr, count);
	temp = sptr;
	for (i=0; i<count; i++)		/* set up array of pointers sorted by value */
	{
		array[i] = temp;
		temp = temp->next;
	}
	return(sptr);
}

/******************************************************************
 *																						*
 *		In-place non-recursive merge sort using label text as key	*
 *																						*
 ******************************************************************/

struct sym *sort_by_name(struct sym *list)
{
	word			i, n;
	struct sym	*a, *b, *todo, *t;

	head_ptr = (struct sym *) malloc(sizeof(struct sym) + 12);
	head_ptr->next = list;
	a = tail_ptr;
	for (n=1; a != head_ptr->next; n = n + n)
	{
		todo = head_ptr->next;
		list = head_ptr;
		while (todo != tail_ptr)
		{
			t = todo;
			a = t;
			for (i=1; i<n; i++)
				t = t->next;
			b = t->next;
			t->next = tail_ptr;
			t = b;
			for (i=1; i<n; i++)
				t = t->next;
			todo = t->next;
			t->next = tail_ptr;
			list->next = merge_by_name(a, b);
			for (i=1; i<=n+n; i++)
				list = list->next;
		}
	}
	return(head_ptr->next);
}

/***************************************************************
 *																					*
 *		In-place non-recursive merge sort using value as key		*
 *																					*
 ***************************************************************/

struct sym *sort_by_value(struct sym *list)
{
	word			i, n;
	struct sym	*a, *b, *todo, *t;

	head_ptr = (struct sym *) malloc(sizeof(struct sym));
	head_ptr->next = list;
	a = tail_ptr;
	for (n=1; a != head_ptr->next; n = n + n)
	{
		todo = head_ptr->next;
		list = head_ptr;
		while (todo != tail_ptr)
		{
			t = todo;
			a = t;
			for (i=1; i<n; i++)
				t = t->next;
			b = t->next;
			t->next = tail_ptr;
			t = b;
			for (i=1; i<n; i++)
				t = t->next;
			todo = t->next;
			t->next = tail_ptr;
			list->next = merge_by_value(a, b);
			for (i=1; i<=n+n; i++)
				list = list->next;
		}
	}
	return(head_ptr->next);
}

/***************************************
 *													*
 *		Merge sub-lists by text field		*
 *													*
 ***************************************/

struct sym *merge_by_name(struct sym *a, struct sym *b)
{
	int			i;
	struct sym	*c;

	c = tail_ptr;
	do
	{

#ifdef	LINUX
		i = strcasecmp(a->name, b->name);
#else
		i = stricmp(a->name, b->name);
#endif
		if (i <= 0)
		{
			c->next = a;
			c = a;
			a = a->next;
		}
		else
		{
			c->next = b;
			c = b;
			b = b->next;
		}
	} while (c != tail_ptr);
	c = tail_ptr->next;
	tail_ptr->next = tail_ptr;
	return(c);
}

/******************************************
 *														*
 *		Merge sub-lists by value field		*
 *														*
 ******************************************/

struct sym *merge_by_value(struct sym *a, struct sym *b)
{
	struct sym	*c;

	c = tail_ptr;
	do
	{
		if (a->val < b->val)
		{
			c->next = a;
			c = a;
			a = a->next;
		}
		else
		{
			c->next = b;
			c = b;
			b = b->next;
		}
	} while (c != tail_ptr);
	c = tail_ptr->next;
	tail_ptr->next = tail_ptr;
	return(c);
}

/******************************************************
 *																		*
 *		Check for redefinitions of label/symbol names	*
 *																		*
 ******************************************************/

void chk_dup_name(struct sym *list, word count)
{
	word	i;

	for (i=0; i<count; i++)
	{

#ifdef	LINUX
		if (!strcasecmp(list->name, list->next->name))
#else
		if (!stricmp(list->name, list->next->name))
#endif
		{
			printf("\nAttempted redefinition of '%s', value 0x%x,"
					 " as value 0x%x\n",
					list->name, list->val, list->next->val);
			exit(USER_ERROR);
		}
		list = list->next;
	}
}

/*********************************************
 *															*
 *		Check for redefinitions of values		*
 *															*
 *********************************************/

void chk_dup_value(struct sym *list, word count)
{
	word	i;

	for (i=0; i<count; i++)
	{
		if (list->val == list->next->val)
		{
			printf("\nAttempted redefinition of value 0x%x, '%s', as '%s'\n",
					list->val, list->name, list->next->name);
			exit(USER_ERROR);
		}
		list = list->next;
	}
}

/***************************************************************
 *																					*
 *							Find symbol or label								*
 *																					*
 *		Binary search using table of pointers to list nodes.		*
 *		This search is based on the fact that:							*
 *					list[first-1]  key < list[last+1]					*
 *		is an invariant. This allows the while loop to involve	*
 *		only one boolean expression, rather than two. The 'if'	*
 *		statement also involves only one boolean expression.		*
 *		The test for equality (has it been found?) is not done	*
 *		until the loop is complete. This significantly speeds		*
 *		up the search compared to a standard binary search.		*
 *																					*
 ***************************************************************/

char *find_entry(word val, word count, SYM_PTR *table)
{
	struct sym	*ptr;
	word			first, mid, last;
	char			*ret;

	first = 1;
	last = count;
	while (first <= last)				/* while more to search... */
	{
		mid = (first + last) >> 1;		/* begin in middle of remaining array */
		ptr = table[mid - 1];			/* get pointer to node */
		if (ptr->val > val)				/* determine which way to go */
			last = mid - 1;				/* search before midpoint */
		else
			first = mid + 1;				/* search after midpoint */
	}
	ret = NULL;								/* assume not found */
	if (last > 0)							/* are we still within the array? */
	{
		ptr = table[last - 1];			/* if so, get last pointer */
		if (val == ptr->val)				/* is it what we're searching for? */
		{
			ptr->used = 1;					/* label/symbol has been used */
			ret = ptr->name;				/* return pointer to caller */
		}
	}
	return(ret);
}

/************************************************************
 *																				*
 *		Allocate new entry in symbol table or label table		*
 *																				*
 ************************************************************/

struct sym *get_smem(int type, int req_size)
{
	struct sym	*ptr;

	ptr = (struct sym*) malloc(sizeof(struct sym) + req_size + 1);
																/* get memory from OS */
	if (ptr == NULL)										/* what? out of memory?... */
	{
		printf("\nINTERNAL ERROR! - No memory for ");
		if (type)
			printf("label");
		else
			printf("symbol");
		printf(" table!\n");
		exit(MEM_ERROR);
	}
	ptr->next = NULL;
	return(ptr);						/* give caller the address */
}

/***************************************
 *													*
 *		Add symbol or label to table		*
 *  Symbol if type == 0, else label.	*
 *													*
 ***************************************/

void add_entry(word val, char *symbol, int type)
{
	struct sym	*nptr, *tbl_ptr;
	char			*cptr;
	char			tbl_name[8];

	cptr = symbol;				/* ensure that input string is null terminated */
	while (*cptr)
	{
		if (!isgraph(*cptr))
			break;
		cptr++;
	}
	*cptr = '\0';

	if (type)
	{
		tbl_ptr = lab_tab;				/* get pointer to correct table */
		strcpy(tbl_name, "label");
		label_count++;
	}
	else
	{
		tbl_ptr = sym_tab;
		strcpy(tbl_name, "symbol");
		symbol_count++;
	}

	nptr = get_smem(type, strlen(symbol));
	if (tbl_ptr == NULL)					/* if first symbol or label... */
	{
		if (type)							/* initialize head pointer */
			lab_tab = nptr;
		else
			sym_tab = nptr;
	}
	else
	{
		if (type)							/* get pointer to end of correct table */
			lab_tab_last->next = nptr;
		else
			sym_tab_last->next = nptr;
	}
	if (type)
		lab_tab_last = nptr;
	else
		sym_tab_last = nptr;
	nptr->used = 0;
	nptr->val = val;						/* set value in new entry */
	strcpy(nptr->name, symbol);		/* and copy text to entry */
}

/*  end of d48.c */

